	#ifndef __CSystemString__
#define __CSystemString__

#include "CString.hpp"
#include "CStringTokeniser.hpp"

//	===========================================================================

using Exponent::Basics::CString;
using Exponent::Basics::CStringTokeniser;

//	===========================================================================

namespace Exponent
{
	namespace Basics
	{
		/**
		 * @class CSystemString CSystemString.hpp
		 * @brief Path handling string class
		 *
		 * Stores a string and provides path handling functionality on that string
		 *
		 * @date 21/08/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.0.1 Fixed bug in the = operator that would turn your string to mush because the string equality was not checking if passed object was this
		 * @version 1.0.2 Added removeLastPathComponent function
		 * @version 1.0.3 Added removeFirstPathComponent function
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CSystemString.hpp,v 1.4 2007/02/08 21:06:44 paul Exp $
		 */
		class CSystemString : public CString
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			const static char *CSYSTEM_STRING_SYSTEM_SEPERATOR;			/**< System specific path seperator */
			const static long CSYSTEM_STRING_MAX_PATH = _MAX_PATH;		/**< Maximum path length */

//	===========================================================================

			/**
			 * Construction
			 * @param string The string to store
			 */
			CSystemString(const char *string = " ");

			/**
			 * Construction
			 * @param string The string to store
			 */
			CSystemString(const CString &string);

			/**
			 * Copy Construction
			 * @param string The string to store
			 */
			CSystemString(const CSystemString &string);

			/**
			 * Destructor
			 */
			virtual ~CSystemString();

//	===========================================================================

			/**
			 * Assignment operator
			 * @param string The string to copy
			 * @retval CSystemString& A reference to this
			 */
			virtual CSystemString &operator = (const CSystemString &string);

			/**
			 * Assignment operator
			 * @param string The string to copy
			 * @retval CSystemString& A reference to this
			 */
			virtual CSystemString &operator = (const char *string);

			/**
			 * Concatenation operator
			 * @param string The string to append to the end of this
			 * @retval CSystemString& A reference to this
			 */
			virtual CSystemString &operator += (const CString &string);

			/**
			 * Concatenation operator
			 * @param string The string to append to the end of this
			 * @retval CSystemString& A reference to this
			 */
			virtual CSystemString &operator += (const char *string);

			/**
			 * Equality operator
			 * @param equals the string to compare to
			 * @retval bool True if strings are equal, false otherwise
			 */
			virtual bool operator == (const char *equals);

			/**
			 * Equality operator
			 * @param string the string to compare to
			 * @retval bool True if strings are equal, false otherwise
			 */
			virtual bool operator == (CSystemString &string);

			/**
			 * InEquality operator
			 * @param string the string to compare to
			 * @retval bool True if strings are inequal, false otherwise
			 */
			virtual bool operator != (const char *string);

			/**
			 * InEquality operator
			 * @param string the string to compare to
			 * @retval bool True if strings are inequal, false otherwise
			 */
			virtual bool operator != (CSystemString &string);

			/**
			 * Check if the string is empty
			 * @retval bool True if number of characters is == 0, false otherwise
			 */
			virtual bool operator ! ();

			/**
			 * Subscript operator
			 * @param index The index of the character you require
			 * @retval char The character at the specified index
			 */
			virtual char operator [] (const long index);

//	===========================================================================

			/**
			 * Does the path have an extension
			 * @retval bool True if path has extension, false otherwise
			 */
			virtual bool hasExtension() const;

			/**
			 * Does the path have this extension
			 * @param ext The extension to check for
			 * @retval bool True if string has matching extension, false otherwise
			 */
			virtual bool hasExtension(const char *ext) const;

			/**
			 * Does the path have this extension
			 * @param ext The extension to check for
			 * @retval bool True if string has matching extension, false otherwise
			 */
			virtual bool hasExtension(const CString &ext) const;

			/**
			 * Append a new extension
			 * @param ext The extension to append. Doesnt require a leading '.'
			 * @param removeOld If true the old extension (if one exists) is remove, if false new extension is simply appended
			 */
			virtual void appendNewExtension(const char *ext, const bool removeOld = true);

			/**
			 * Append a new extension
			 * @param ext The extension to append. Doesnt require a leading '.'
			 * @param removeOld If true the old extension (if one exists) is remove, if false new extension is simply appended
			 */
			virtual void appendNewExtension(const CString &ext, const bool removeOld = true);

			/**
			 * Remove the extension from the string
			 */
			virtual void removeExtension();

			/**
			 * Is this a folder
			 * @retval bool True if path is to a folder, false otherwise
			 */
			virtual bool isFolder() const;

			/**
			 * Strip the filename from the string
			 */
			virtual void stripFilename();

			/**
			 * Get the folder part of the path
			 * @retval CString The folder part of the path. 
			 */
			virtual CString getFolderPath();

			/**
			 * Append a new system seperator (\ or / depending on OS)
			 */
			virtual void appendSystemSeperator();

			/**
			 * Append a path string - before appending, appends a path seperator
			 * @param path The stirng to append to the end of the path
			 */
			virtual void appendPath(const CString &path);

			/**
			 * Get the drive letter of the string (on mac volume name)
			 * @param withDriveSeperators If true includes ":\\" (PC) or "/" (Mac), if false doesnt include them
			 * @retval CString The drive letter
			 */
			virtual CString getDriveLetter(bool withDriveSeperators = true);

			/**
			 * Get the extension
			 * @retval CString  The extension, up to you to delete memory returned
			 */
			virtual CString getTheExtension();

			/**
			 * Get the filename
			 * @retval CString The filename, up to you to delete memory returned
			 */
			virtual CString getTheFilename() const;

			/**
			 * Get the number of path components
			 * @retval long The number of path components
			 */
			virtual long getNumberOfPathComponents();

			/**
			 * Get a specific path component
			 * @param index The index of the path component
			 * @retval CString* The path component. You are responsible for deleting the returned memory
			 */
			virtual CString *getPathComponent(const long index);

			/**
			 * Remove the last path component
			 */
			virtual void removeLastPathComponent();

			/**
			 * Remove the first path component
			 */
			virtual void removeFirstPathComponent();

			/**
			 * Replace all path seperators with the correct format for the current OS
			 */
			virtual void replacePathSeperators();

#ifndef WIN32
			/**
			 * Get as a mac FSSPec (path must exist to be created sucessfully)
			 * @retval FSSpec* The mac FSSpec
			 */
			FSSpec *getStringAsFSSpec() const;

			/**
			 * Get as a mac FSRef (path must exist to be created sucessfully)
			 * @retval FSRef* The mac FSRef
			 */
			FSRef *getStringAsFSRef() const;
#endif

//	===========================================================================
		};
	}
}
#endif	// End of CSystemString.hpp